---
name: skill-creator-guide
description: Guide Claude through creating valid custom skills with proper YAML frontmatter, file structure, and naming conventions to pass all validation checks on first attempt
---

# Skill Creator Guide

## Overview

This skill teaches Claude how to create valid custom skills for Claude.ai that pass all validation requirements on the first upload attempt. It covers YAML frontmatter rules, file structure, naming conventions, and common pitfalls.

**When to use this skill:**
- User asks to create a custom skill
- User mentions uploading a skill to Claude
- User references SKILL.md or skills directory
- User wants to package instructions for Claude to follow

## Critical Validation Rules

### YAML Frontmatter Requirements

**REQUIRED FORMAT:**
```yaml
---
name: skill-name-here
description: Clear description of what this skill does and when to use it
---
```

**STRICT RULES:**
1. **name field:**
   - ONLY lowercase letters, numbers, and hyphens
   - NO uppercase letters
   - NO spaces
   - NO underscores
   - NO special characters except hyphens
   - Example VALID: `my-skill-name`, `skill-123`, `data-processor`
   - Example INVALID: `My Skill`, `my_skill`, `MySkill`, `my.skill`

2. **description field:**
   - Maximum 200 characters
   - Should clearly explain when Claude should use this skill
   - Must be specific enough for Claude to know when to invoke it
   - Single line (no line breaks in the description)

3. **Optional fields** (only add if needed):
   - `version:` - Track versions (e.g., `1.0.0`)
   - `dependencies:` - Required packages (e.g., `python>=3.8, pandas>=1.5.0`)

4. **Forbidden in YAML frontmatter:**
   - NO Unicode characters (✅ ❌ → ✓ etc.)
   - NO emojis
   - NO special quotes (" " instead of " ")
   - Only standard ASCII characters

### File Structure

**Correct structure:**
```
skill-name/
├── SKILL.md          (required)
├── REFERENCE.md      (optional - supplemental info)
├── scripts/          (optional)
│   └── helper.py
└── resources/        (optional)
    └── template.xlsx
```

**When creating ZIP:**
```
skill-name.zip
└── skill-name/       <- folder must be at root of ZIP
    └── SKILL.md
```

**WRONG structure that will fail:**
```
skill-name.zip
└── SKILL.md          <- files directly in ZIP root (FAILS)
```

### Markdown Body Guidelines

After the YAML frontmatter, the markdown body should:

1. **Start with clear sections:**
   - `## Overview` - Explain what the skill does
   - `## When to Use` - Be specific about triggering conditions
   - `## Instructions` - Step-by-step guidelines

2. **Use standard markdown:**
   - Headers: `##`, `###`
   - Lists: `-` or `1.`
   - Code blocks: triple backticks
   - Bold: `**text**`
   - Italic: `*text*`

3. **Avoid in markdown body:**
   - Excessive Unicode symbols (some are OK, but keep minimal)
   - Complex formatting that might break parsing
   - External links that might fail

## Step-by-Step Skill Creation Process

### Step 1: Understand the Requirement

Ask yourself:
- What specific task should this skill help with?
- When should Claude automatically invoke this skill?
- What instructions does Claude need?

### Step 2: Choose the Skill Name

Apply the naming rules:
- Convert spaces to hyphens: "Brand Guidelines" → `brand-guidelines`
- Make lowercase: "MySkill" → `myskill`
- Replace special chars: "My_Skill!" → `my-skill`

### Step 3: Write the Description

Make it specific and actionable:
- BAD: "Helps with documents"
- GOOD: "Apply company brand guidelines to all presentations and documents, including colors, fonts, and logo usage"

Description should answer: "When should Claude use this skill?"

### Step 4: Create YAML Frontmatter

```yaml
---
name: your-skill-name
description: Specific description of what this skill does and when to use it
---
```

**Checklist before proceeding:**
- [ ] Name is all lowercase
- [ ] Name uses only letters, numbers, hyphens
- [ ] Description is under 200 characters
- [ ] Description explains when to use the skill
- [ ] No Unicode symbols in YAML section
- [ ] Three dashes before and after

### Step 5: Write the Markdown Body

**Template structure:**
```markdown
# Skill Name (Human-Friendly Title)

## Overview
Brief explanation of what this skill does and its purpose.

## When to Use This Skill
- Specific scenario 1
- Specific scenario 2
- Specific scenario 3

## Core Instructions

### Section 1: Main Guidelines
Detailed instructions here...

### Section 2: Examples
Examples of proper usage...

### Section 3: Best Practices
- Best practice 1
- Best practice 2

## Do's and Don'ts

### DO
- Specific action 1
- Specific action 2

### DON'T
- Thing to avoid 1
- Thing to avoid 2

## Usage Note
Final notes about when and how to apply this skill.
```

### Step 6: Create Directory Structure

```bash
mkdir skill-name
cd skill-name
# Create SKILL.md with content
# Add any additional files if needed
```

### Step 7: Create ZIP File

**On Mac/Linux:**
```bash
zip -r skill-name.zip skill-name/
```

**On Windows:**
Right-click folder → "Send to" → "Compressed (zipped) folder"

**Verify ZIP structure:**
- Unzip and check that the folder is at root level
- SKILL.md should be inside the folder, not at ZIP root

### Step 8: Validate Before Upload

**Pre-upload checklist:**
- [ ] Skill name uses only lowercase, numbers, hyphens
- [ ] Description under 200 characters
- [ ] YAML frontmatter has exactly 3 dashes before and after
- [ ] No Unicode symbols in YAML section
- [ ] ZIP contains folder at root (not files directly)
- [ ] SKILL.md is inside the folder
- [ ] All referenced files exist in the ZIP

## Common Errors and Solutions

### Error: "unexpected key in SKILL.md frontmatter"

**Cause:** Unrecognized field in YAML or Unicode characters
**Solution:** 
- Remove any optional fields you added
- Stick to only `name:` and `description:`
- Check for Unicode symbols (✅ → ✓) and replace

### Error: "Skill name can only contain lowercase letters, numbers, and hyphens"

**Cause:** Uppercase letters, spaces, or special characters in name
**Solution:**
```
"My Skill" → "my-skill"
"My_Skill" → "my-skill"
"MySkill!" → "myskill"
```

### Error: "ZIP file structure is incorrect"

**Cause:** Files at ZIP root instead of in folder
**Solution:**
- Ensure folder structure: `skill-name.zip/skill-name/SKILL.md`
- Re-create ZIP with folder included

## Examples

### Example 1: Simple Style Guide Skill

```yaml
---
name: company-writing-style
description: Apply company writing style guidelines for professional communications, including tone, vocabulary, and formatting preferences
---

# Company Writing Style

## Overview
This skill applies our company's writing style to all communications.

## When to Use
- Writing emails
- Creating presentations
- Drafting reports
- Any professional communication

## Style Guidelines

### Tone
- Professional but approachable
- Clear and concise
- Avoid jargon

### Formatting
- Use bullet points for lists
- Keep paragraphs short
- Bold key terms

## Examples
[Provide specific examples here]
```

### Example 2: Data Processing Skill

```yaml
---
name: data-processor
description: Process and analyze CSV data files with specific company formatting rules and generate summary reports
---

# Data Processor Skill

## Overview
Processes CSV files according to company standards.

## When to Use
- User uploads CSV file
- User asks for data analysis
- User requests formatted report

## Processing Steps
1. Load the CSV file
2. Validate column names
3. Clean data
4. Generate summary statistics
5. Create formatted output

[Additional instructions...]
```

## Best Practices Summary

1. **Keep it focused**: One skill, one clear purpose
2. **Be specific in description**: Claude needs to know when to invoke
3. **Use simple names**: lowercase-with-hyphens
4. **Test locally first**: Create and review before uploading
5. **Start minimal**: Add complexity only if needed
6. **Document clearly**: Future users should understand immediately
7. **Version your skills**: Update version field when making changes

## Quick Reference Card

**Valid Skill Name Characters:** `a-z`, `0-9`, `-`
**YAML Frontmatter:** Name + Description only (optional: version, dependencies)
**Max Description Length:** 200 characters
**File Structure:** ZIP contains folder, folder contains SKILL.md
**Character Encoding:** ASCII only in YAML, UTF-8 OK in markdown body

## Workflow for Creating Skills

1. **Define purpose** → What task does this solve?
2. **Name it** → lowercase-with-hyphens
3. **Describe it** → When should Claude use this?
4. **Write YAML** → Validate format
5. **Write instructions** → Clear and structured
6. **Create directory** → Proper structure
7. **ZIP it** → Folder at root
8. **Validate** → Check all rules
9. **Upload** → Should work first time!

---

## Usage Note

When a user asks to create a skill, follow this guide exactly:
1. Start with the naming rules
2. Create proper YAML frontmatter
3. Structure the markdown body clearly
4. Create proper directory structure
5. Generate ZIP file correctly
6. Provide validation checklist

This systematic approach ensures skills pass validation on first upload attempt.
